# bvh_importer.py

bl_info = {
    "name": "Moqapnet",
    "blender": (4, 1, 0),
    "category": "3DMotaas",
    "version": (1, 2, 0),
    "author": "Algolysis Ltd.",
    "description": "Import BVH files into Blender and process via API",
    "location": "View3D > Sidebar > MoqapLens",
}

import bpy
import os
import tempfile
import requests
from requests.exceptions import RequestException
from bpy_extras.io_utils import ImportHelper
from urllib.parse import urlparse

# Module-level poll state for non-blocking polling (retarget)
_poll_state = {
    'active': False,
    'task_id': None,
    'attempts': 0,
    'max_attempts': 60,
    'interval': 3,
    'timer_registered': False,
}

# Module-level poll state for search
_search_poll_state = {
    'active': False,
    'task_id': None,
    'attempts': 0,
    'max_attempts': 60,
    'interval': 3,
    'timer_registered': False,
    'is_keyword_search': False,
}

def _get_api_base():
    parsed = urlparse("https://search.moqap.eu")
    return f"{parsed.scheme}://{parsed.netloc}"

def _derive_status_url(endpoint, task_id):
    parsed = urlparse(endpoint)
    base = f"{parsed.scheme}://{parsed.netloc}"
    return f"{base}/status/{task_id}"

def _derive_download_url(endpoint, task_id):
    parsed = urlparse(endpoint)
    base = f"{parsed.scheme}://{parsed.netloc}"
    return f"{base}/download/{task_id}"

def _derive_search_url():
    return f"{_get_api_base()}/search"

def _derive_search_json_url(task_id):
    return f"{_get_api_base()}/get_search_json/{task_id}"

def _derive_result_motion_url(segment_hash):
    return f"{_get_api_base()}/get_result_motion/{segment_hash}"

def _derive_original_motion_url(file_hash):
    return f"{_get_api_base()}/get_original_motion/?hash={file_hash}"

def _get_addon_prefs():
    addon_key = __package__ if __package__ else __name__
    return bpy.context.preferences.addons[addon_key].preferences

# ---------------- Armature BVH path helpers ----------------
def set_armature_bvh_path(obj: bpy.types.Object, path: str) -> bool:
    try:
        if not obj or obj.type != 'ARMATURE':
            return False
        obj['bvh_source_path'] = os.path.abspath(path)
        if getattr(obj, 'data', None):
            try:
                obj.data['bvh_source_path'] = os.path.abspath(path)
            except Exception:
                pass
        return True
    except Exception:
        return False

# ---------------- HTTP helpers ----------------
def _http_post_multipart(url, file_field_name, filename, file_bytes, form_fields=None, headers=None, timeout=120):
    files = {file_field_name: (filename, file_bytes, 'application/octet-stream')}
    data = form_fields if form_fields else None
    try:
        resp = requests.post(url, files=files, data=data, headers=headers or {}, timeout=timeout)
    except RequestException as e:
        raise RuntimeError(f"Network error: {e}") from e

    ct = resp.headers.get('Content-Type', '')
    if 'application/json' in ct or resp.headers.get('Content-Type', '').startswith('application/json'):
        try:
            return resp.json()
        except Exception as e:
            raise RuntimeError(f"Failed to decode JSON response: {e}") from e
    return {"result_bvh_raw": resp.content}

def _import_bvh(filepath, report=None):
    try:
        bpy.ops.import_anim.bvh(filepath=filepath)
        if report:
            report({'INFO'}, f"Imported processed BVH: {os.path.basename(filepath)}")
        return True
    except Exception as e:
        if report:
            report({'ERROR'}, f"Failed to import processed BVH: {e}")
        return False

# ------------- Properties ----------------
class SearchResultItem(bpy.types.PropertyGroup):
    """Stores a single search result"""
    hash: bpy.props.StringProperty(name="Hash", default="")
    name: bpy.props.StringProperty(name="Name", default="")
    labels: bpy.props.StringProperty(name="Labels", default="")
    parent_name: bpy.props.StringProperty(name="Parent Name", default="")
    parent_hash: bpy.props.StringProperty(name="Parent Hash", default="")
    start_frame: bpy.props.IntProperty(name="Start Frame", default=0)
    end_frame: bpy.props.IntProperty(name="End Frame", default=0)
    distance: bpy.props.FloatProperty(name="Distance", default=0.0)
    rank: bpy.props.IntProperty(name="Rank", default=0)
    is_keyword_result: bpy.props.BoolProperty(name="Is Keyword Result", default=False)

class MotionSearchProperties(bpy.types.PropertyGroup):
    # Retarget state
    last_task_id: bpy.props.StringProperty(name="Last Task ID", default="")
    polling_active: bpy.props.BoolProperty(name="Polling Active", default=False)
    poll_attempts: bpy.props.IntProperty(name="Poll Attempts", default=0)
    last_result_path: bpy.props.StringProperty(name="Last Result Path", default="")

    # Search state
    search_query: bpy.props.StringProperty(name="Search", description="Keywords to search for", default="")
    search_bvh_path: bpy.props.StringProperty(name="Search BVH", description="BVH file to search with", default="", subtype='FILE_PATH')
    search_results: bpy.props.CollectionProperty(type=SearchResultItem)
    search_results_index: bpy.props.IntProperty(name="Selected Result", default=0)
    search_active: bpy.props.BoolProperty(name="Search Active", default=False)
    search_attempts: bpy.props.IntProperty(name="Search Attempts", default=0)


# ------------- UI List for search results ------------------
class MOQAPLENS_UL_search_results(bpy.types.UIList):
    bl_idname = "MOQAPLENS_UL_search_results"

    def draw_item(self, context, layout, data, item, icon, active_data, active_property, index):
        if self.layout_type in {'DEFAULT', 'COMPACT'}:
            row = layout.row(align=True)
            if item.is_keyword_result:
                # Keyword search: show name and labels
                row.label(text=f"{item.rank}. {item.name}")
            else:
                # BVH search: show parent_name and frame range
                row.label(text=f"{item.rank}. {item.parent_name}")
                row.label(text=f"[{item.start_frame}-{item.end_frame}]")
        elif self.layout_type == 'GRID':
            layout.label(text=item.name if item.is_keyword_result else item.parent_name)

# ------------- UI Panel ------------------
class BVHMotionSearchPanel(bpy.types.Panel):
    bl_label = "MoqapLens"
    bl_idname = "VIEW3D_PT_motion_search"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'MoqapLens'

    def draw(self, context):
        layout = self.layout
        props = context.scene.motion_search_props

        # === Search Section ===
        box = layout.box()
        box.label(text="Search", icon='VIEWZOOM')

        # Keyword search
        box.label(text="Search with keyword:")
        box.prop(props, "search_query", text="")

        # Or separator
        row = box.row()
        row.alignment = 'CENTER'
        row.label(text="— or —")

        # BVH file search
        box.label(text="Search with BVH:")
        box.prop(props, "search_bvh_path", text="")

        # Single search button
        box.operator("wm.motion_search", text="Search", icon='VIEWZOOM')

        # Search status
        if props.search_active:
            box.label(text=f"Searching... ({props.search_attempts})", icon='TIME')

        # Results list
        if len(props.search_results) > 0:
            box.template_list(
                "MOQAPLENS_UL_search_results", "",
                props, "search_results",
                props, "search_results_index",
                rows=4
            )
            box.operator("wm.import_search_result", text="Import", icon='IMPORT')

        layout.separator()

        # === Retarget Section ===
        box = layout.box()
        box.label(text="Retarget", icon='ARMATURE_DATA')

        # Import BVH
        box.operator("import_scene.bvh_custom", text="Import BVH", icon='IMPORT')

        # Show active armature info
        active_obj = context.view_layer.objects.active
        active_path = None
        if active_obj and active_obj.type == 'ARMATURE':
            active_path = active_obj.get('bvh_source_path', '')

        if active_path:
            box.label(text=f"Source: {os.path.basename(active_path)}")
            box.operator("wm.process_bvh_api", text="Retarget", icon='PLAY')

            # Status
            if props.polling_active:
                box.label(text=f"Processing... ({props.poll_attempts})", icon='TIME')
            elif props.last_result_path:
                box.label(text="Done", icon='CHECKMARK')
        else:
            box.label(text="Select armature with BVH", icon='INFO')

# ------------- Operators -----------------
class ImportBVHOperator(bpy.types.Operator, ImportHelper):
    bl_idname = "import_scene.bvh_custom"
    bl_label = "Import BVH"
    bl_description = "Import a BVH file into the Blender scene"
    bl_options = {"REGISTER", "UNDO"}

    filename_ext = ".bvh"
    filter_glob: bpy.props.StringProperty(default="*.bvh", options={"HIDDEN"})

    def execute(self, context):
        filepath = self.filepath
        try:
            bpy.ops.import_anim.bvh(filepath=filepath)
            # Attach the source to selected or active armature
            try:
                for sel in context.selected_objects:
                    if sel.type == 'ARMATURE':
                        set_armature_bvh_path(sel, filepath)
                if not any(o.type == 'ARMATURE' for o in context.selected_objects):
                    a = context.view_layer.objects.active
                    if a and a.type == 'ARMATURE':
                        set_armature_bvh_path(a, filepath)
            except Exception:
                pass
            self.report({'INFO'}, f"Successfully imported {os.path.basename(filepath)}")
        except Exception as e:
            self.report({'ERROR'}, f"Failed to import {os.path.basename(filepath)}: {str(e)}")
        return {'FINISHED'}

# Process current BVH via API
class WM_OT_process_bvh_api(bpy.types.Operator):
    """Send the BVH to the retargeting API and import the processed result"""
    bl_idname = "wm.process_bvh_api"
    bl_label = "Process BVH via API"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        props = context.scene.motion_search_props

        try:
            addon_prefs = _get_addon_prefs()
        except Exception:
            self.report({'ERROR'}, "Could not load add-on preferences. Make sure the add-on is enabled.")
            return {'CANCELLED'}

        endpoint = getattr(addon_prefs, "api_endpoint", "") or "https://search.moqap.eu/retarget"
        api_key = getattr(addon_prefs, "api_key", "")

        bvh_path = None
        try:
            active_obj = context.view_layer.objects.active
            if active_obj and active_obj.type == 'ARMATURE':
                bvh_path = active_obj.get('bvh_source_path')
        except Exception:
            bvh_path = None

        if not bvh_path or not os.path.exists(bvh_path):
            self.report({'ERROR'}, "No BVH path found on the active armature.")
            return {'CANCELLED'}

        try:
            with open(bvh_path, "rb") as f:
                bvh_bytes = f.read()
        except Exception as e:
            self.report({'ERROR'}, f"Failed to read BVH: {e}")
            return {'CANCELLED'}

        headers = {"accept": "application/json"}
        if api_key:
            headers["Authorization"] = f"Bearer {api_key}"

        try:
            self.report({'INFO'}, "Uploading BVH to retargeting API...")
            resp = _http_post_multipart(
                endpoint,
                file_field_name="file",
                filename=os.path.basename(bvh_path),
                file_bytes=bvh_bytes,
                headers=headers,
                timeout=120
            )
        except Exception as e:
            self.report({'ERROR'}, f"API call failed: {e}")
            return {'CANCELLED'}

        task_id = None
        if isinstance(resp, dict) and "task_id" in resp:
            task_id = str(resp["task_id"])
        elif isinstance(resp, dict) and "id" in resp:
            task_id = str(resp["id"])
        else:
            self.report({'ERROR'}, "API response did not include 'task_id'.")
            return {'CANCELLED'}

        props.last_task_id = task_id

        # Start polling for result
        _poll_state['active'] = True
        _poll_state['task_id'] = task_id
        _poll_state['attempts'] = 0
        _poll_state['max_attempts'] = 60
        _poll_state['interval'] = 3
        props.polling_active = True
        props.poll_attempts = 0
        if not _poll_state.get('timer_registered'):
            bpy.app.timers.register(_poll_timer, first_interval=3)
            _poll_state['timer_registered'] = True

        self.report({'INFO'}, f"Processing... Task ID: {task_id}")
        return {'FINISHED'}

# Search operator
class WM_OT_motion_search(bpy.types.Operator):
    """Search for motions using keywords or BVH file"""
    bl_idname = "wm.motion_search"
    bl_label = "Search Motions"

    def execute(self, context):
        props = context.scene.motion_search_props

        # Determine search mode: BVH file takes priority if provided
        bvh_path = props.search_bvh_path.strip()
        query = props.search_query.strip()

        use_file = bvh_path and os.path.exists(bvh_path)

        if not use_file and not query:
            self.report({'WARNING'}, "Enter keywords or select a BVH file")
            return {'CANCELLED'}

        # Clear previous results
        props.search_results.clear()

        try:
            addon_prefs = _get_addon_prefs()
            api_key = getattr(addon_prefs, "api_key", "")
        except Exception:
            api_key = ""

        headers = {"accept": "application/json"}
        if api_key:
            headers["Authorization"] = f"Bearer {api_key}"

        search_url = _derive_search_url()

        if use_file:
            # BVH file search
            try:
                with open(bvh_path, "rb") as f:
                    bvh_bytes = f.read()
            except Exception as e:
                self.report({'ERROR'}, f"Failed to read BVH: {e}")
                return {'CANCELLED'}

            try:
                files = {"file": (os.path.basename(bvh_path), bvh_bytes, 'application/octet-stream')}
                resp = requests.post(
                    search_url,
                    files=files,
                    headers=headers,
                    timeout=60
                )
                resp.raise_for_status()
                data = resp.json()
            except Exception as e:
                self.report({'ERROR'}, f"Search request failed: {e}")
                return {'CANCELLED'}
        else:
            # Keyword search
            try:
                resp = requests.post(
                    search_url,
                    data={"keywords": query},
                    headers=headers,
                    timeout=30
                )
                resp.raise_for_status()
                data = resp.json()
            except Exception as e:
                self.report({'ERROR'}, f"Search request failed: {e}")
                return {'CANCELLED'}

        task_id = data.get("task_id") or data.get("id")
        if not task_id:
            self.report({'ERROR'}, "No task_id returned from search")
            return {'CANCELLED'}

        # Start polling for search results
        _search_poll_state['active'] = True
        _search_poll_state['task_id'] = task_id
        _search_poll_state['attempts'] = 0
        _search_poll_state['max_attempts'] = 60
        _search_poll_state['interval'] = 2
        _search_poll_state['is_keyword_search'] = not use_file
        props.search_active = True
        props.search_attempts = 0

        if not _search_poll_state.get('timer_registered'):
            bpy.app.timers.register(_search_poll_timer, first_interval=2)
            _search_poll_state['timer_registered'] = True

        self.report({'INFO'}, f"Searching... Task ID: {task_id}")
        return {'FINISHED'}

# Import selected search result
class WM_OT_import_search_result(bpy.types.Operator):
    """Import the selected motion from search results"""
    bl_idname = "wm.import_search_result"
    bl_label = "Import Search Result"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        props = context.scene.motion_search_props

        if len(props.search_results) == 0:
            self.report({'WARNING'}, "No search results")
            return {'CANCELLED'}

        idx = props.search_results_index
        if idx < 0 or idx >= len(props.search_results):
            self.report({'WARNING'}, "Invalid selection")
            return {'CANCELLED'}

        result = props.search_results[idx]
        motion_hash = result.hash
        is_keyword = result.is_keyword_result

        try:
            addon_prefs = _get_addon_prefs()
            api_key = getattr(addon_prefs, "api_key", "")
        except Exception:
            api_key = ""

        headers = {}
        if api_key:
            headers["Authorization"] = f"Bearer {api_key}"

        # Use different endpoint based on search type
        if is_keyword:
            motion_url = _derive_original_motion_url(motion_hash)
        else:
            motion_url = _derive_result_motion_url(motion_hash)

        try:
            resp = requests.get(motion_url, headers=headers, timeout=60)
            resp.raise_for_status()
        except Exception as e:
            self.report({'ERROR'}, f"Failed to fetch motion: {e}")
            return {'CANCELLED'}

        # Save to temp file
        tmp_dir = tempfile.gettempdir()
        out_path = os.path.join(tmp_dir, f"motion_{motion_hash[:8]}.bvh")
        with open(out_path, 'wb') as f:
            f.write(resp.content)

        # Import the BVH
        try:
            bpy.ops.import_anim.bvh(filepath=out_path)
        except Exception as e:
            self.report({'ERROR'}, f"Failed to import BVH: {e}")
            return {'CANCELLED'}

        # Set frame range only for BVH search results (keyword results don't have frame info)
        if not is_keyword and result.start_frame and result.end_frame:
            context.scene.frame_start = result.start_frame
            context.scene.frame_end = result.end_frame
            context.scene.frame_current = result.start_frame

        if is_keyword:
            self.report({'INFO'}, f"Imported {result.name}")
        else:
            self.report({'INFO'}, f"Imported {result.parent_name} [{result.start_frame}-{result.end_frame}]")
        return {'FINISHED'}

# ------------- Poll timer (status endpoint) -------------
def _poll_timer():
    if not _poll_state.get('active'):
        _poll_state['timer_registered'] = False
        return None

    task_id = _poll_state.get('task_id')
    attempts = _poll_state.get('attempts', 0)
    max_attempts = _poll_state.get('max_attempts', 60)
    interval = _poll_state.get('interval', 3)

    _poll_state['attempts'] = attempts + 1

    try:
        scene = bpy.context.scene
        props = scene.motion_search_props
        props.poll_attempts = _poll_state['attempts']
        props.polling_active = True
    except Exception:
        pass

    try:
        addon_prefs = _get_addon_prefs()
    except Exception:
        _poll_state['active'] = False
        _poll_state['timer_registered'] = False
        return None

    endpoint = getattr(addon_prefs, 'api_endpoint', '') or "https://search.moqap.eu/retarget"
    api_key = getattr(addon_prefs, 'api_key', '')
    headers = {"accept": "application/json"}
    if api_key:
        headers['Authorization'] = f"Bearer {api_key}"

    status_url = _derive_status_url(endpoint, task_id)
    try:
        resp = requests.get(status_url, headers=headers, timeout=10)
        resp.raise_for_status()
    except Exception:
        if _poll_state['attempts'] >= max_attempts:
            _poll_state['active'] = False
            _poll_state['timer_registered'] = False
            return None
        return interval

    try:
        j = resp.json()
    except Exception:
        if _poll_state['attempts'] >= max_attempts:
            _poll_state['active'] = False
            _poll_state['timer_registered'] = False
            return None
        return interval

    # Check task status - the API returns status like "SUCCESS", "PENDING", "STARTED", etc.
    status_value = str(j.get('status', '')).upper()

    # Task is complete when status is SUCCESS
    if status_value == 'SUCCESS':
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.polling_active = False
            props.poll_attempts = _poll_state['attempts']
        except Exception:
            pass

        # Download the result from /download/{task_id}
        try:
            download_url = _derive_download_url(endpoint, task_id)
            download_headers = {}
            if api_key:
                download_headers['Authorization'] = f"Bearer {api_key}"

            r2 = requests.get(download_url, headers=download_headers, timeout=60)
            r2.raise_for_status()

            # Save the downloaded BVH to temp directory
            tmp_dir = tempfile.gettempdir()
            out_path = os.path.join(tmp_dir, f"retarget_{task_id}.bvh")
            with open(out_path, 'wb') as f:
                f.write(r2.content)

            try:
                scene = bpy.context.scene
                props = scene.motion_search_props
                props.last_result_path = out_path
            except Exception:
                pass

            _import_bvh(out_path, report=None)
        except Exception:
            pass

        _poll_state['active'] = False
        _poll_state['timer_registered'] = False
        return None

    # Task failed
    if status_value == 'FAILURE':
        _poll_state['active'] = False
        _poll_state['timer_registered'] = False
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.polling_active = False
        except Exception as e:
            print(e)
            pass
        return None

    # Task still pending/running - continue polling
    if _poll_state['attempts'] >= max_attempts:
        _poll_state['active'] = False
        _poll_state['timer_registered'] = False
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.polling_active = False
        except Exception:
            pass
        return None

    return interval

# ------------- Search poll timer -------------
def _search_poll_timer():
    if not _search_poll_state.get('active'):
        _search_poll_state['timer_registered'] = False
        return None

    task_id = _search_poll_state.get('task_id')
    attempts = _search_poll_state.get('attempts', 0)
    max_attempts = _search_poll_state.get('max_attempts', 60)
    interval = _search_poll_state.get('interval', 2)

    _search_poll_state['attempts'] = attempts + 1

    try:
        scene = bpy.context.scene
        props = scene.motion_search_props
        props.search_attempts = _search_poll_state['attempts']
        props.search_active = True
    except Exception:
        pass

    try:
        addon_prefs = _get_addon_prefs()
        api_key = getattr(addon_prefs, 'api_key', '')
    except Exception:
        api_key = ""

    headers = {"accept": "application/json"}
    if api_key:
        headers['Authorization'] = f"Bearer {api_key}"

    # Check status first
    status_url = _derive_status_url("https://search.moqap.eu", task_id)
    try:
        resp = requests.get(status_url, headers=headers, timeout=10)
        resp.raise_for_status()
        status_data = resp.json()
    except Exception:
        if _search_poll_state['attempts'] >= max_attempts:
            _search_poll_state['active'] = False
            _search_poll_state['timer_registered'] = False
            try:
                scene = bpy.context.scene
                props = scene.motion_search_props
                props.search_active = False
            except Exception:
                pass
            return None
        return interval

    status_value = str(status_data.get('status', '')).upper()

    if status_value == 'SUCCESS':
        # Fetch search results
        try:
            search_json_url = _derive_search_json_url(task_id)
            resp2 = requests.get(search_json_url, headers=headers, timeout=30)
            resp2.raise_for_status()
            data = resp2.json()
        except Exception:
            _search_poll_state['active'] = False
            _search_poll_state['timer_registered'] = False
            try:
                scene = bpy.context.scene
                props = scene.motion_search_props
                props.search_active = False
            except Exception:
                pass
            return None

        # Parse and store results
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.search_active = False
            props.search_results.clear()

            is_keyword = _search_poll_state.get('is_keyword_search', False)
            results = data.get('results', [])
            for r in results:
                item = props.search_results.add()
                item.hash = r.get('hash', '')
                item.is_keyword_result = is_keyword
                if is_keyword:
                    # Keyword search: use name and labels
                    item.name = r.get('name', 'Unknown')
                    item.labels = r.get('labels', '')
                else:
                    # BVH search: use parent_name and frame info
                    item.parent_name = r.get('parent_name', r.get('name', 'Unknown'))
                    item.parent_hash = r.get('parent_hash', '')
                    item.start_frame = r.get('start_frame', 0)
                    item.end_frame = r.get('end_frame', 0)
                    item.distance = r.get('distance', 0.0)
                item.rank = r.get('rank', 0)
        except Exception:
            pass

        _search_poll_state['active'] = False
        _search_poll_state['timer_registered'] = False
        return None

    if status_value == 'FAILURE':
        _search_poll_state['active'] = False
        _search_poll_state['timer_registered'] = False
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.search_active = False
        except Exception:
            pass
        return None

    # Still pending - continue polling
    if _search_poll_state['attempts'] >= max_attempts:
        _search_poll_state['active'] = False
        _search_poll_state['timer_registered'] = False
        try:
            scene = bpy.context.scene
            props = scene.motion_search_props
            props.search_active = False
        except Exception:
            pass
        return None

    return interval

# ------------- Registration ----------------
def menu_func_import(self, context):
    self.layout.operator(ImportBVHOperator.bl_idname, text="BVH (.bvh)")

def register():
    bpy.utils.register_class(SearchResultItem)
    bpy.utils.register_class(MotionSearchProperties)
    bpy.utils.register_class(MOQAPLENS_UL_search_results)
    bpy.utils.register_class(BVHMotionSearchPanel)
    bpy.utils.register_class(ImportBVHOperator)
    bpy.utils.register_class(WM_OT_process_bvh_api)
    bpy.utils.register_class(WM_OT_motion_search)
    bpy.utils.register_class(WM_OT_import_search_result)
    bpy.types.TOPBAR_MT_file_import.append(menu_func_import)
    bpy.types.Scene.motion_search_props = bpy.props.PointerProperty(type=MotionSearchProperties)

def unregister():
    bpy.types.TOPBAR_MT_file_import.remove(menu_func_import)
    del bpy.types.Scene.motion_search_props
    bpy.utils.unregister_class(WM_OT_import_search_result)
    bpy.utils.unregister_class(WM_OT_motion_search)
    bpy.utils.unregister_class(WM_OT_process_bvh_api)
    bpy.utils.unregister_class(ImportBVHOperator)
    bpy.utils.unregister_class(BVHMotionSearchPanel)
    bpy.utils.unregister_class(MOQAPLENS_UL_search_results)
    bpy.utils.unregister_class(MotionSearchProperties)
    bpy.utils.unregister_class(SearchResultItem)

if __name__ == "__main__":
    register()
